window.onload = function() {
    // Define projections
    proj4.defs("EPSG:32724", "+proj=utm +zone=24 +south +datum=WGS84 +units=m +no_defs");
    proj4.defs("EPSG:4674", "+proj=longlat +datum=WGS84 +no_defs");
    ol.proj.proj4.register(proj4);

    // WMS base layer from sisprom
    const wmsBaseLayer = new ol.layer.Tile({
        title: 'Jequie Base',
        source: new ol.source.TileWMS({
            url: 'https://sisprom-br.com.br:8443/geoserver/sisprom/wms',
            params: {
                'LAYERS': 'sisprom:jequie-02',
                'VERSION': '1.1.0',
                'FORMAT': 'image/png',
                'TILED': true
            },
            serverType: 'geoserver',
            projection: 'EPSG:32724'
        })
    });

    // Google Satellite layer
    const satelliteLayer = new ol.layer.Tile({
        title: 'Google Satellite',
        source: new ol.source.XYZ({
            url: 'https://mt1.google.com/vt/lyrs=s&x={x}&y={y}&z={z}',
            maxZoom: 20
        }),
        visible: false
    });

    // Street overlay layer
    const streetOverlay = new ol.layer.Tile({
        title: 'Street Overlay',
        source: new ol.source.TileWMS({
            url: 'https://sisprom-br.com.br:8443/geoserver/sisprom/wms',
            params: {
                'LAYERS': 'sisprom:ruas_jequie_2023_wgs84',
                'VERSION': '1.1.0',
                'FORMAT': 'image/png',
                'TILED': true
            },
            serverType: 'geoserver',
            projection: 'EPSG:32724'
        }),
        visible: true
    });

    // Cadastro overlay layer with labels
    const cadastroOverlay = new ol.layer.Tile({
        title: 'Cadastro Overlay',
        source: new ol.source.TileWMS({
            url: 'https://sisprom-br.com.br:8443/geoserver/sisprom/wms',
            params: {
                'LAYERS': 'sisprom:cadastrojeque_2000',
                'VERSION': '1.1.1',
                'FORMAT': 'image/png',
                'TILED': true,
                'TRANSPARENT': true,
                'STYLES': 'label_lotes',  // Enable lot number labels
                'format_options': 'layout:message',
                'enable_text': true
            },
            serverType: 'geoserver',
            projection: 'EPSG:32724'  // Changed to match map projection
        }),
        visible: true,  // Set to true by default
        minZoom: 16     // Adjusted zoom level for better visibility
    });

    // Create highlight layer for selected features
    const highlightSource = new ol.source.Vector();
    const highlightLayer = new ol.layer.Vector({
        source: highlightSource,
        style: new ol.style.Style({
            stroke: new ol.style.Stroke({
                color: '#0000FF',
                width: 1,
                opacity: 1
            }),
            fill: new ol.style.Fill({
                color: 'rgba(0, 0, 255, 0.05)'
            })
        })
    });

    // Create map
    const map = new ol.Map({
        target: 'map',
        layers: [wmsBaseLayer, satelliteLayer, streetOverlay, cadastroOverlay, highlightLayer],
        view: new ol.View({
            projection: 'EPSG:32724',
            center: [384704.79805, 8467929.13975],
            zoom: 16,
            maxZoom: 20,
            minZoom: 14
        })
    });

    // Add pointer cursor on hover
    map.on('pointermove', function(evt) {
        if (evt.dragging || !cadastroOverlay.getVisible()) {
            return;
        }

        const pixel = map.getEventPixel(evt.originalEvent);
        const hit = map.hasFeatureAtPixel(pixel);
        map.getTargetElement().style.cursor = hit ? 'pointer' : '';
    });

    // Create layer switcher control
    const layerSwitcher = document.createElement('div');
    layerSwitcher.className = 'layer-switcher';

    // Base layer controls
    const baseLayerDiv = document.createElement('div');
    baseLayerDiv.innerHTML = '<h4>Base Layers</h4>';

    const wmsRadio = document.createElement('input');
    wmsRadio.type = 'radio';
    wmsRadio.name = 'base';
    wmsRadio.checked = true;
    wmsRadio.onchange = () => {
        wmsBaseLayer.setVisible(true);
        satelliteLayer.setVisible(false);
    };

    const satelliteRadio = document.createElement('input');
    satelliteRadio.type = 'radio';
    satelliteRadio.name = 'base';
    satelliteRadio.onchange = () => {
        wmsBaseLayer.setVisible(false);
        satelliteLayer.setVisible(true);
    };

    baseLayerDiv.appendChild(wmsRadio);
    baseLayerDiv.appendChild(document.createTextNode(' WMS Base'));
    baseLayerDiv.appendChild(document.createElement('br'));
    baseLayerDiv.appendChild(satelliteRadio);
    baseLayerDiv.appendChild(document.createTextNode(' Satellite'));

    // Overlay controls
    const overlayDiv = document.createElement('div');
    overlayDiv.innerHTML = '<h4>Overlays</h4>';

    const streetCheck = document.createElement('input');
    streetCheck.type = 'checkbox';
    streetCheck.checked = true;
    streetCheck.onchange = (e) => {
        streetOverlay.setVisible(e.target.checked);
    };

    const cadastroCheck = document.createElement('input');
    cadastroCheck.type = 'checkbox';
    cadastroCheck.checked = false;
    cadastroCheck.onchange = (e) => {
        cadastroOverlay.setVisible(e.target.checked);
    };

    overlayDiv.appendChild(streetCheck);
    overlayDiv.appendChild(document.createTextNode(' Streets'));
    overlayDiv.appendChild(document.createElement('br'));
    overlayDiv.appendChild(cadastroCheck);
    overlayDiv.appendChild(document.createTextNode(' Cadastro'));

    layerSwitcher.appendChild(baseLayerDiv);
    layerSwitcher.appendChild(overlayDiv);

    map.getViewport().appendChild(layerSwitcher);

    // Add scale line
    map.addControl(new ol.control.ScaleLine());

    // Create sidebar
    const sidebar = document.createElement('div');
    sidebar.className = 'sidebar';
    sidebar.innerHTML = `
        <div class="sidebar-header">
            <h3>Informações do Lote</h3>
            <button class="close-btn">&times;</button>
        </div>
        <div class="sidebar-content">
            <div class="feature-info"></div>
        </div>
    `;
    document.body.appendChild(sidebar);

    // Function to highlight feature
    function highlightFeature(feature) {
        highlightSource.clear();
        if (feature) {
            // Transform feature geometry to map projection
            const featureGeom = new ol.format.GeoJSON().readFeature(feature);
            featureGeom.getGeometry().transform('EPSG:4674', 'EPSG:32724');
            highlightSource.addFeature(featureGeom);

            // Get feature extent and zoom to it
            const extent = featureGeom.getGeometry().getExtent();
            map.getView().fit(extent, {
                padding: [100, 100, 100, 100],
                maxZoom: 20,
                duration: 500
            });
        }
    }

    // Add click handler
    map.on('singleclick', function(evt) {
        if (!cadastroOverlay.getVisible()) return;

        const viewResolution = map.getView().getResolution();
        const url = cadastroOverlay.getSource().getFeatureInfoUrl(
            evt.coordinate,
            viewResolution,
            'EPSG:32724',
            {'INFO_FORMAT': 'application/json'}
        );

        if (url) {
            fetch(url)
                .then(response => response.json())
                .then(data => {
                    if (data.features && data.features.length > 0) {
                        const feature = data.features[0];
                        const properties = feature.properties;

                        // Highlight the selected feature
                        highlightFeature(feature);

                        // Display feature info in sidebar
                        const featureInfo = sidebar.querySelector('.feature-info');
                        featureInfo.innerHTML = Object.entries(properties)
                            .map(([key, value]) => `
                                <div class="feature-info-item">
                                    <span class="feature-info-label">${key}</span>
                                    <span class="feature-info-value">${value}</span>
                                </div>
                            `)
                            .join('');

                        sidebar.classList.add('active');
                    }
                });
        }
    });

    // Add close button handler
    const closeBtn = sidebar.querySelector('.close-btn');
    closeBtn.onclick = function() {
        sidebar.classList.remove('active');
        highlightSource.clear(); // Remove highlight when closing sidebar
    };
}