// ===== Inicializa o mapa =====
var map = L.map('map').setView([-13.8607, -40.0786], 14);

// ===== Camada base =====
var baseLayer = L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
    attribution: '&copy; OpenStreetMap contributors'
}).addTo(map);

// ===== Layers para separar situações =====
var wfsLayer = L.geoJSON(null, { onEachFeature: onEachWFSFeature }).addTo(map);
var corretoLayer = L.geoJSON(null);
var atualizarLayer = L.geoJSON(null);

// ===== Estilos por Situação =====
function getCorPorSituacao(situacao) {
    switch (situacao) {
        case 'CORRETO': return '#00FF00';
        case 'ATUALIZAR': return '#FF0000';
        case 'REGULARIZADO': return '#0080FF';
        case 'NOTIFICADO': return '#FFFF00';
        case 'DISTRIBUIDO': return '#FF00FF';
        default: return '#000000';
    }
}

// ===== Função para carregar WFS =====
function loadWFS() {
    fetch('https://sisprom-br.com.br:8443/geoserver/sisprom/wfs?service=WFS&version=2.0.0&request=GetFeature&typeNames=sisprom:cadastrotributos&outputFormat=application/json&srsName=EPSG:4326')
        .then(response => response.json())
        .then(data => {
            wfsLayer.addData(data);

            data.features.forEach(feature => {
                if (feature.properties.situacao === "CORRETO") {
                    corretoLayer.addData(feature);
                } else if (feature.properties.situacao === "ATUALIZAR") {
                    atualizarLayer.addData(feature);
                }
            });

            console.log("Dados WFS carregados.");
        })
        .catch(error => console.error('Erro ao carregar WFS:', error));
}

// ===== Ações por Feature =====
function onEachWFSFeature(feature, layer) {
    const inscricao = feature.properties.inscricao;

    if (!inscricao) {
        console.warn("Inscrição não encontrada:", feature);
        return;
    }

    const situacaoAtual = feature.properties.situacao;

    layer.setStyle({
        color: getCorPorSituacao(situacaoAtual),
        weight: 2,
        opacity: 1,
        fillOpacity: 0.4
    });

    layer.on('click', function () {
        const userGroup = sessionStorage.getItem('usergroupids'); // obtido da sessão no PHP
        const isGroup7 = userGroup === '7';
        const isGroup5 = userGroup === '5';

        let options = `
            <option value="CORRETO" ${situacaoAtual === 'CORRETO' ? 'selected' : ''}>CORRETO</option>
            <option value="ATUALIZAR" ${situacaoAtual === 'ATUALIZAR' ? 'selected' : ''}>ATUALIZAR</option>
            <option value="REGULARIZADO" ${situacaoAtual === 'REGULARIZADO' ? 'selected' : ''}>REGULARIZADO</option>
            <option value="NOTIFICADO" ${situacaoAtual === 'NOTIFICADO' ? 'selected' : ''}>NOTIFICADO</option>
            <option value="DISTRIBUIDO" ${situacaoAtual === 'DISTRIBUIDO' ? 'selected' : ''}>DISTRIBUÍDO</option>
        `;

        // Se for grupo 5, limita os valores
        if (isGroup5) {
            options = `
                <option value="REGULARIZADO" ${situacaoAtual === 'REGULARIZADO' ? 'selected' : ''}>REGULARIZADO</option>
                <option value="NOTIFICADO" ${situacaoAtual === 'NOTIFICADO' ? 'selected' : ''}>NOTIFICADO</option>
                <option value="DISTRIBUIDO" ${situacaoAtual === 'DISTRIBUIDO' ? 'selected' : ''}>DISTRIBUÍDO</option>
            `;
        }

        let fiscalSelectHTML = '';
        if (isGroup7) {
            fiscalSelectHTML = `
                <label for="fiscal_select_${inscricao}">Fiscal responsável:</label>
                <select id="fiscal_select_${inscricao}"></select><br><br>
            `;
        }

        const popupContent = `
            <b>Inscrição:</b> ${inscricao}<br>
            <b>Situação:</b><br>
            <select id="select_${inscricao}">
                ${options}
            </select><br><br>
            ${fiscalSelectHTML}
            <button onclick="salvarSituacao('${inscricao}')">Salvar Situação</button>
        `;

        layer.bindPopup(popupContent).openPopup();

        // Só carrega fiscais se for grupo 7
        if (isGroup7) {
            fetch('get_fiscais.php')
                .then(response => response.json())
                .then(data => {
                    const select = document.getElementById(`fiscal_select_${inscricao}`);
                    data.forEach(fiscal => {
                        const option = document.createElement('option');
                        option.value = fiscal.username;
                        option.text = fiscal.name;
                        select.appendChild(option);
                    });
                });
        }
    });
}

// ===== Salvar Situação =====
function salvarSituacao(inscricao) {
    const select = document.getElementById(`select_${inscricao}`);
    if (!select) {
        alert("Erro: campo de seleção não encontrado.");
        return;
    }

    const novaSituacao = select.value;

    console.log('Inscrição:', inscricao);
    console.log('Nova Situação:', novaSituacao);

    // Atualiza localmente
    wfsLayer.eachLayer(function (layer) {
        const props = layer.feature.properties;
        if (props.inscricao === inscricao) {
            props.situacao = novaSituacao;
            const novaCor = getCorPorSituacao(novaSituacao);
            layer.setStyle({
                color: novaCor,
                fillColor: novaCor,
                weight: 2,
                opacity: 1,
                fillOpacity: 0.4
            });
        }
    });

    // Envia para PHP
    fetch('vendor/geojequie/salvar_situacao.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
            inscricao: inscricao,
            situacao: novaSituacao
        })
    })
        .then(response => response.json())
        .then(data => {
            if (data.status === 'ok') {
                alert('Situação salva com sucesso!');
            } else {
                alert('Erro ao salvar: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Erro:', error);
            alert('Erro de comunicação.');
        });
}

function carregarFiscaisParaPopup(selectId) {
    fetch('vendor/geojequie/get_fiscais.php')
        .then(res => res.json())
        .then(res => {
            if (res.status === 'ok') {
                const select = document.getElementById(selectId);
                select.innerHTML = '<option value="">Selecione</option>';
                res.data.forEach(fiscal => {
                    const opt = document.createElement('option');
                    opt.value = fiscal.username;
                    opt.textContent = fiscal.nome + ' (' + fiscal.username + ')';
                    select.appendChild(opt);
                });
            }
        })
        .catch(err => {
            console.error('Erro ao carregar fiscais:', err);
        });
}

// ===== Iniciar =====
loadWFS();

// ===== PrintMap =====
var printerControl = L.easyPrint({
    tileLayer: baseLayer,
    sizeModes: ['Current', 'A4Landscape', 'A4Portrait'],
    filename: 'mapa_impresso',
    exportOnly: false,
    hideControlContainer: true
}).addTo(map);

function manualPrint() {
    printer.printMap('CurrentSize', 'MyManualPrint')
}