const F = {
  inscricao:'inscricao', name:'name', numero_doc:'numero_doc', bairro:'bairro', logradouro:'logradouro', quadra:'quadra', lote:'lote',
  numero_lot:'numero_lot', unidade:'unidade', utilizacao:'utilizacao', area_terre:'area_terre', area_unida:'area_unida', longitude:'longitude',
  latitude:'latitude', situacao:'situacao', zoneamento:'zoneamento', setor:'setor'
};
const thematicFields = [
  {value:'setor',label:'Setor'},
  {value:'bairro',label:'Bairro'},
  {value:'zoneamento',label:'Zoneamento'},
  {value:'situacao',label:'Situação'},
  {value:'utilizacao',label:'Utilização'}
];

const wfsUrl = "https://sisprom-br.com.br:8443/geoserver/sisprom/wfs?"+
  "service=WFS&version=2.0.0&request=GetFeature&typeNames=sisprom:Jequie_2025_postgis"+
  "&outputFormat=application/json&srsName=EPSG:3857";

// Dados globais
let features = [];
let setorOpts = new Set(), bairroOpts = new Set(), zoneamentoOpts = new Set();
let tableState = {limit:10, verMais:false};
let selectedFeatures = [];
let debounceTimer = null;

// OpenLayers
const vectorSource = new ol.source.Vector();
const vectorLayer = new ol.layer.Vector({
  source: vectorSource,
  style: feature => {
    if(window.thematicOn && window.thematicField){
      let field = window.thematicField;
      let val = feature.get(field)||'N/D';
      let color = thematicColor(val, field);
      let isSelected = selectedFeatures.some(f=>f.get(F.inscricao)===feature.get(F.inscricao));
      return new ol.style.Style({
        fill: new ol.style.Fill({color: isSelected ? '#fffadc' : color}),
        stroke: new ol.style.Stroke({color: isSelected ? '#ffae00' : '#3476e1', width: isSelected?3:1.5})
      });
    }
    let isSelected = selectedFeatures.some(f=>f.get(F.inscricao)===feature.get(F.inscricao));
    return new ol.style.Style({
      fill: new ol.style.Fill({color: isSelected ? '#fffadc' : '#e6f3ff'}),
      stroke: new ol.style.Stroke({color: isSelected ? '#ffae00' : '#3476e1', width: isSelected?3:1.2})
    });
  }
});
const map = new ol.Map({
  target: 'map',
  layers: [new ol.layer.Tile({ source: new ol.source.OSM() }), vectorLayer],
  view: new ol.View({ center: ol.proj.fromLonLat([-40.0786,-13.8607]), zoom: 14 })
});

// Utilitário para criar opções de select
function setSelectOptions(selectId, values) {
  document.getElementById(selectId).innerHTML =
    `<option value="">Todos</option>` + [...values].sort().map(v=>`<option>${v}</option>`).join('');
}

// Pega dados e inicia tudo
fetch(wfsUrl)
  .then(r=>r.json())
  .then(data=>{
    const format = new ol.format.GeoJSON();
    features = format.readFeatures(data, { dataProjection:'EPSG:3857', featureProjection:'EPSG:3857' });
    features.forEach(f=>{
      setorOpts.add(f.get(F.setor));
      bairroOpts.add(f.get(F.bairro));
      zoneamentoOpts.add(f.get(F.zoneamento));
    });
    updateKPIs();
    updateFilterOptions();
    updateAll();
  });

function updateKPIs(){
  document.getElementById('kpi-total').textContent = features.length;
  document.getElementById('kpi-setores').textContent = setorOpts.size;
  document.getElementById('kpi-bairros').textContent = bairroOpts.size;
  document.getElementById('kpi-zoneamento').textContent = zoneamentoOpts.size;
}

// Debounce busca para performance
document.getElementById('busca').oninput = function() {
  clearTimeout(debounceTimer);
  debounceTimer = setTimeout(updateAll, 240);
};

function updateFilterOptions(){
  function setFiltroValores(){
    let campo = document.getElementById('filtro-campo').value;
    setSelectOptions('filtro-valor', new Set(features.map(f=>f.get(F[campo])).filter(Boolean)));
  }
  setFiltroValores();
  document.getElementById('filtro-campo').onchange = ()=>{setFiltroValores(); updateAll();};
  document.getElementById('filtro-valor').onchange = updateAll;
  document.getElementById('cmp-grupo').onchange=()=>updateCorrecoes();
  document.getElementById('tabela-limite').onchange=()=>{
    tableState.limit = parseInt(document.getElementById('tabela-limite').value)||10;
    tableState.verMais = false;
    updateTable(filteredFeatures());
  };
}

function filteredFeatures(){
  const campo = document.getElementById('filtro-campo').value;
  const val = document.getElementById('filtro-valor').value;
  const q = (document.getElementById('busca').value||"").toLowerCase();
  return features.filter(f=>
    (!val||f.get(F[campo])===val) &&
    (!q || ( (f.get(F.name)||"").toLowerCase().includes(q) ||
             (f.get(F.inscricao)||"").toLowerCase().includes(q) ||
             (f.get(F.logradouro)||"").toLowerCase().includes(q) ) )
  );
}

// Temático
window.thematicOn = false;
window.thematicField = null;
function thematicColor(val, field){
  let pal = ['#e57373','#7986cb','#64b5f6','#4db6ac','#fff176','#ffd54f','#ba68c8','#a1887f','#90a4ae','#ff8a65','#388e3c','#c62828','#01579b','#e67c73'];
  let opts = [...new Set(features.map(f=>f.get(field)||'N/D'))].sort();
  let idx = opts.indexOf(val);
  return pal[idx % pal.length] + '99';
}
document.getElementById('btn-thematic').onclick = ()=>{
  let field = prompt("Mapa temático: escolha campo\n(setor, bairro, zoneamento, situacao, utilizacao)", window.thematicField||"setor");
  field = field?.toLowerCase();
  if(!thematicFields.some(f=>f.value===field)){ alert('Campo inválido.'); return;}
  window.thematicField = field;
  window.thematicOn = true;
  updateAll();
  document.getElementById('btn-thematic').textContent = "Temático: "+(thematicFields.find(f=>f.value===field).label||field);
  document.getElementById('btn-thematic').style.background='#38b6a1';
  document.getElementById('btn-thematic').title = "Clique duplo para desativar o temático.";
};
document.getElementById('btn-thematic').ondblclick = ()=>{
  window.thematicOn = false; window.thematicField = null;
  updateAll();
  document.getElementById('btn-thematic').textContent = "Mapa Temático";
  document.getElementById('btn-thematic').style.background='#2266e2';
  document.getElementById('btn-thematic').title = "Clique para ativar mapa temático.";
};

function updateAll(){
  const feats = filteredFeatures();
  selectedFeatures.length = 0;
  vectorSource.clear(); vectorSource.addFeatures(feats);
  updateCharts(feats);
  updateCorrecoes(feats);
  updateTable(feats);
}

function groupCount(list, field){
  const stat={}; list.forEach(f=>{
    let v = f.get(field)||'N/D'; stat[v]=stat[v]?stat[v]+1:1;
  }); return stat;
}

// Função util para criar/atualizar um gráfico ECharts (evita memory leak)
function updateEChart(dom, option) {
  let chart = echarts.getInstanceByDom(dom);
  if (chart) chart.dispose();
  chart = echarts.init(dom);
  chart.setOption(option);
  return chart;
}

function updateCharts(feats){
  let bairroStats = groupCount(feats, F.bairro);
  let bairros = Object.keys(bairroStats), vals = bairros.map(b=>bairroStats[b]);
  updateEChart(document.getElementById('chart-bar-bairro'), {
    tooltip:{}, grid:{left:36,right:14,bottom:26,top:24,containLabel:true},
    xAxis:{type:'category',data:bairros,axisLabel:{fontSize:10,rotate:16}}, yAxis:{type:'value',minInterval:1},
    series:[{data:vals,type:'bar',barWidth:'60%',itemStyle:{color:'#1976d2',borderRadius:[5,5,0,0]}}]
  });

  let usoStats = groupCount(feats, F.utilizacao);
  let usos = Object.keys(usoStats), usosVals = usos.map(u=>usoStats[u]);
  updateEChart(document.getElementById('chart-pie-uso'), {
    tooltip:{},
    legend:{
      orient: window.innerWidth>900 ? 'vertical' : 'horizontal',
      left: window.innerWidth>900 ? '78%' : 'center',
      top: window.innerWidth>900 ? 'middle' : 'bottom',
      itemWidth:16, itemHeight:16, textStyle:{fontSize:12},
      data: usos
    },
    color:['#1976d2','#ffb64c','#7b1fa2','#43c888','#e9516a','#5ac8fa','#ffa59b','#b0b1ec','#789262','#ba67ea'],
    series:[{
      name:'Utilização',type:'pie',radius:['44%','75%'],
      center: window.innerWidth>900 ? ['41%','50%'] : ['50%','50%'],
      label:{formatter:'{b}: {c}'},
      data:usos.map((u,i)=>({value:usosVals[i],name:u}))
    }]
  });

  let sitStats = groupCount(feats, F.situacao);
  let sits = Object.keys(sitStats), sitsVals = sits.map(s=>sitStats[s]);
  updateEChart(document.getElementById('chart-pie-situacao'), {
    tooltip:{}, legend:{bottom:0}, color:['#59b4fa','#e9516a','#ffb64c','#4caf50'],
    series:[{name:'Situação',type:'pie',radius:['44%','70%'],label:{formatter:'{b}: {c}'},data:sits.map((u,i)=>({value:sitsVals[i],name:u}))}]
  });
}

function updateCorrecoes(feats){
  if(!feats) feats = filteredFeatures();
  const grupo = document.getElementById('cmp-grupo').value;
  const situacoes = Array.from(new Set(feats.map(f=>f.get(F.situacao)||'N/D')));
  const agrupadores = Array.from(new Set(feats.map(f=>f.get(F[grupo])||'N/D'))).sort();
  let data = situacoes.map(sit => ({
    name: sit,
    type: 'bar',
    stack: 'total',
    label: {show: false},
    emphasis: {focus: 'series'},
    data: agrupadores.map(g =>
      feats.filter(f => (f.get(F[grupo])===g) && (f.get(F.situacao)===sit)).length
    )
  }));
  updateEChart(document.getElementById('chart-cmp'), {
    tooltip: { trigger: 'axis', axisPointer: {type: 'shadow'} },
    legend: {data: situacoes, bottom: 0},
    grid: {left:40,right:8,bottom:40,top:24,containLabel:true},
    xAxis: {type: 'category', data: agrupadores, axisLabel:{fontSize:10,rotate:16}},
    yAxis: {type: 'value', minInterval:1},
    series: data
  });
}

// ------------------- TABELA E SELEÇÃO -------------------

function updateTable(feats){
  // Se houver seleção de feições, mostra só elas
  if(selectedFeatures.length > 0){
    feats = feats.filter(f => selectedFeatures.some(sel => sel.get(F.inscricao) === f.get(F.inscricao)));
  }
  let limit = tableState.verMais ? feats.length : (parseInt(document.getElementById('tabela-limite').value) || 10);
  let total = feats.length;
  let btn = document.getElementById('btnVerMais');
  let fields = [F.inscricao,F.name,F.bairro,F.logradouro,F.situacao,F.zoneamento,F.setor,F.utilizacao,F.area_terre];

  document.getElementById('status-selecao').innerHTML =
    selectedFeatures.length > 0 ? `Mostrando apenas <b>${selectedFeatures.length}</b> imóvel(is) selecionado(s) no mapa.` : '';

  let btnClear = document.getElementById('btnClearSel');
  if(selectedFeatures.length>0){
    btnClear.style.display='inline-block';
    btnClear.onclick = function(){
      selectedFeatures.length=0; updateTable(filteredFeatures()); vectorLayer.changed();
    };
  }else{
    btnClear.style.display='none';
  }
  if(total===0){
    document.getElementById('tableArea').innerHTML="<em>Nenhum registro encontrado.</em>";
    document.getElementById('tabela-total').textContent='';
    btn.style.display='none';
    return;
  }
  let html = '<table class="data-table"><tr>'+
    fields.map(f=>`<th>${f.charAt(0).toUpperCase()+f.slice(1)}</th>`).join('')+'</tr>';
  feats.slice(0,limit).forEach((f,i)=>{
    let isSelected = selectedFeatures.some(sel => sel.get(F.inscricao) === f.get(F.inscricao));
    html += `<tr data-inscr="${f.get(F.inscricao)||''}" class="${isSelected?'table-multiselect':''}">`+
      fields.map(k=>`<td>${f.get(k)||""}</td>`).join('')+"</tr>";
  });
  html += "</table>";
  // Botão exportar CSV
  html += `<button class="vermais-btn" style="margin-top:10px" onclick="exportTableToCSV()">Exportar CSV</button>`;
  document.getElementById('tableArea').innerHTML = html;

  if(total>limit){
    document.getElementById('tabela-total').textContent = `Mostrando ${limit} de ${total}`;
    btn.style.display='inline-block';
    btn.textContent='Ver mais';
  }else if(total>0 && tableState.verMais){
    document.getElementById('tabela-total').textContent = `Mostrando todos (${total})`;
    btn.style.display='inline-block';
    btn.textContent='Ver menos';
  }else{
    document.getElementById('tabela-total').textContent = `Total: ${total}`;
    btn.style.display='none';
  }
  // Seleção ao clicar na linha
  document.querySelectorAll('#tableArea tr[data-inscr]').forEach(tr=>{
    tr.onclick = function(e){
      let inscr = this.getAttribute('data-inscr');
      let feat = feats.find(f=>f.get(F.inscricao)===inscr);
      if(!feat) return;
      if(e.ctrlKey||e.metaKey){
        let already = selectedFeatures.some(f=>f.get(F.inscricao)===inscr);
        if(already){
          selectedFeatures = selectedFeatures.filter(f=>f.get(F.inscricao)!==inscr);
        }else{
          selectedFeatures.push(feat);
        }
      }else{
        selectedFeatures = [feat];
      }
      updateTable(filteredFeatures());
      map.getView().fit(feat.getGeometry().getExtent(),{maxZoom:17,padding:[40,40,40,40]});
      vectorLayer.changed();
    };
  });
}
document.getElementById('btnVerMais').onclick=function(){
  tableState.verMais = !tableState.verMais;
  updateTable(filteredFeatures());
}

// Exporta tabela como CSV
window.exportTableToCSV = function() {
  let rows = Array.from(document.querySelectorAll('.data-table tr'));
  let csv = rows.map(tr => Array.from(tr.children).map(td=>`"${(td.innerText||'').replace(/"/g,'""')}"`).join(',')).join('\n');
  let blob = new Blob([csv], {type: 'text/csv'});
  let url = URL.createObjectURL(blob);
  let a = document.createElement('a');
  a.href = url; a.download = 'imoveis.csv';
  document.body.appendChild(a); a.click(); document.body.removeChild(a);
  URL.revokeObjectURL(url);
}

// ------------------- MAPA E SELEÇÃO -------------------
document.body.onmousedown = function(e){ window._lastMouseEvent = e; };
map.on('singleclick', function(evt) {
  let featsClicked = [];
  map.forEachFeatureAtPixel(evt.pixel, function(feature) {
    featsClicked.push(feature);
  });
  let mouseEvt = (evt.originalEvent || window._lastMouseEvent || {});
  if(featsClicked.length){
    if(mouseEvt.ctrlKey || mouseEvt.metaKey){
      featsClicked.forEach(f=>{
        let already = selectedFeatures.some(sel=>sel.get(F.inscricao)===f.get(F.inscricao));
        if(!already) selectedFeatures.push(f);
      });
    }else{
      selectedFeatures = [featsClicked[0]];
    }
    updateTable(filteredFeatures());
    vectorLayer.changed();
    let inscr = featsClicked[0].get(F.inscricao);
    let tr = document.querySelector(`#tableArea tr[data-inscr="${inscr}"]`);
    tr?.scrollIntoView({behavior:'smooth',block:'center'});
  }else{
    selectedFeatures.length=0;
    updateTable(filteredFeatures());
    vectorLayer.changed();
  }
});

// Responsividade dos charts
window.addEventListener('resize',()=>{
  ['chart-bar-bairro','chart-pie-uso','chart-pie-situacao','chart-cmp'].forEach(id=>{
    echarts.getInstanceByDom(document.getElementById(id))?.resize();
  });
  updateCharts(filteredFeatures());
});

