<?php

use Dompdf\Dompdf;
use Dompdf\Options;

class calcularValorMercadoForm extends TPage
{
    private $form;

    /* =========================
       HELPERS (URL WFS + cURL)
       ========================= */
    private function buildWfsUrl(string $inscricao): string
    {
        // saneia inscrição (apenas dígitos e X/x)
        $inscricao = preg_replace('/[^0-9Xx]/', '', (string)$inscricao);

        $base = 'https://sisprom-br.com.br:8443/geoserver/sisprom/wfs';
        $params = [
            'service'      => 'WFS',
            'version'      => '2.0.0',
            'request'      => 'GetFeature',
            'typeNames'    => 'sisprom:Jequie_2025_postgis',
            'outputFormat' => 'application/json',
            'srsName'      => 'EPSG:4674',
            // IMPORTANTE: aspas ficam no valor; http_build_query faz o encode
            'CQL_FILTER'   => "inscricao='" . addslashes($inscricao) . "'",
        ];
        return $base . '?' . http_build_query($params, '', '&', PHP_QUERY_RFC3986);
    }

    private function fetchUrl(string $url): string
    {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_CONNECTTIMEOUT => 8,
            CURLOPT_TIMEOUT        => 15,
            CURLOPT_SSL_VERIFYPEER => true, // se cert do :8443 estiver incompleto, teste TEMPORARIAMENTE como false
            CURLOPT_SSL_VERIFYHOST => 2,
            CURLOPT_HTTPHEADER     => ['Accept: application/json'],
        ]);
        $body = curl_exec($ch);
        $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $err  = curl_error($ch);
        curl_close($ch);

        if ($body === false) {
            throw new Exception("Falha ao consultar WFS: $err (HTTP $http)");
        }
        if ($http < 200 || $http >= 300) {
            throw new Exception("WFS respondeu HTTP $http\nURL=$url\nResposta (início): " . substr($body, 0, 400));
        }
        if ($body === '' || $body === null) {
            throw new Exception('WFS retornou vazio');
        }
        return $body;
    }

    // Mantém assinatura antiga para compatibilidade
    private function getUrlContent($url)
    {
        // Redireciona para a versão robusta com cURL
        return $this->fetchUrl($url);
    }

    public function __construct()
    {
        parent::__construct();
        parent::setTargetContainer('adianti_right_panel');

        // use um único lugar para definir o modo
        $modo = $_GET['modo'] ?? 'consulta';

        $this->form = new BootstrapFormBuilder('form_enable_disable');
        $this->form->setFormTitle('Avaliação Imobiliária');
        $this->form->setFieldSizes('100%');
        $this->form->setProperty('style', 'margin-bottom:20px');

        // Centraliza marcador no mapa ao abrir (se lat/lng vierem por GET)
        if (!empty($_GET['latitude']) && !empty($_GET['longitude'])) {
            $lat = (float) $_GET['latitude'];
            $lng = (float) $_GET['longitude'];
            TScript::create("setMapPoint($lat, $lng);");
        }

        // === Campos principais ===
        $id         = new THidden('id');
        $code       = new TEntry('code');
        $zoneamento = new TEntry('zoneamento');
        $tipo       = new TCombo('tipo');
        $padrao     = new TCombo('padrao');
        $area_unida = new TEntry('area_unida');
        $valor      = new TEntry('valor');

        $lote       = new TEntry('lote');
        $quadra     = new TEntry('quadra');
        $setor      = new TEntry('setor');
        $bairro     = new TEntry('bairro');
        $numero_lot = new TEntry('numero_lot');
        $logradouro = new TEntry('logradouro');
        $name       = new TEntry('name');
        $latitude   = new TEntry('latitude');
        $longitude  = new TEntry('longitude');

        // Configs
        $valor->setEditable(false);
        $valor->style = 'font-weight:bold; color:green; background:#f0fff0';

        $tipo->addItems([
            'residencial'   => 'Residencial',
            'comercial'     => 'Comercial',
            'industrial'    => 'Industrial',
            'institucional' => 'Institucional',
        ]);

        $padrao->addItems([
            'baixo' => 'Baixo',
            'medio' => 'Médio',
            'alto'  => 'Alto',
            'luxo'  => 'Luxo',
        ]);

        // Grupo 1
        $this->form->addFields([$id]);
        $this->form->addContent([new TFormSeparator('Dados do Lote', '#333')]);
        $this->form->addFields([new TLabel('Inscrição')], [$code]);
        $this->form->addFields([new TLabel('Lote')], [$lote], [new TLabel('Quadra')], [$quadra]);
        $this->form->addFields([new TLabel('Setor')], [$setor], [new TLabel('Nº do Lote')], [$numero_lot]);
        $this->form->addFields([new TLabel('Logradouro')], [$logradouro], [new TLabel('Bairro')], [$bairro]);
        $this->form->addFields([new TLabel('Proprietário')], [$name]);
        $this->form->addFields([new TLabel('Latitude')], [$latitude]);
        $this->form->addFields([new TLabel('Longitude')], [$longitude]);

        // Grupo 2
        $this->form->addContent([new TFormSeparator('Parâmetros de Avaliação', '#555')]);
        $this->form->addFields([new TLabel('Zoneamento')], [$zoneamento]);
        $this->form->addFields([new TLabel('Área construída (m²)')], [$area_unida]);
        $this->form->addFields([new TLabel('Tipo de uso')], [$tipo]);
        $this->form->addFields([new TLabel('Padrão construtivo')], [$padrao]);

        // Resultado
        $this->form->addContent([new TFormSeparator('Resultado Estimado', '#2a7d2e')]);
        $this->form->addFields([new TLabel('Valor estimado (R$)')], [$valor]);

        // Ações
        if ($modo === 'visualizar') {
            foreach ([$code, $zoneamento, $area_unida, $tipo, $padrao, $valor,
                      $lote, $quadra, $setor, $bairro, $numero_lot,
                      $logradouro, $name, $latitude, $longitude] as $field) {
                $field->setEditable(false);
            }
        } else {
            $this->form->addAction('Calcular', new TAction([$this, 'onCalcular']), 'fa:calculator green');
            $this->form->addAction('Salvar Avaliação', new TAction([$this, 'onSave']), 'fa:save blue');
        }
        $this->form->addAction('Gerar PDF', new TAction([$this, 'onGeneratePdf']), 'fa:file-pdf-o red');
        $this->form->addAction('Fechar', new TAction([$this, 'onClose']), 'fa:times red');

        // === Dados do lote via WFS (por inscrição) ===
        $inscricao = $_GET['inscricao'] ?? null;
        $formData  = [];

        if ($inscricao) {
            try {
                $url    = $this->buildWfsUrl($inscricao);
                $json   = $this->fetchUrl($url);
                $geojson = json_decode($json);

                if (!isset($geojson->features) || count($geojson->features) === 0) {
                    new TMessage('warning', "Nenhuma feição encontrada para a inscrição {$inscricao}.");
                } else {
                    $props = $geojson->features[0]->properties ?? null;

                    if ($props) {
                        $formData = [
                            'code'       => $props->inscricao  ?? '',
                            'name'       => $props->name       ?? '',
                            'zoneamento' => $props->zoneamento ?? '',
                            'lote'       => $props->lote       ?? '',
                            'quadra'     => $props->quadra     ?? '',
                            'setor'      => $props->setor      ?? '',
                            'bairro'     => $props->bairro     ?? '',
                            'numero_lot' => $props->numero_lot ?? '',
                            'logradouro' => $props->logradouro ?? '',
                            // normaliza caso venha com vírgula
                            'area_unida' => isset($props->area_unida) ? str_replace(',', '.', (string)$props->area_unida) : '',
                            'latitude'   => $props->latitude   ?? '',
                            'longitude'  => $props->longitude  ?? '',
                        ];

                        // Posição no mapa (apenas uma chamada)
                        if (isset($props->latitude, $props->longitude) && $props->latitude !== '' && $props->longitude !== '') {
                            $lat = (float)$props->latitude;
                            $lng = (float)$props->longitude;
                            TScript::create("setMapPoint($lat, $lng);");
                        }

                        // Se for visualizar/edição: busca avaliação salva
                        if (in_array($modo, ['visualizar', 'edicao'], true)) {
                            $urlApi    = "https://geo-el02.cloud.sisprom-br.com.br/vendor/geojequie/buscar_avaliacoes.php?inscricao={$inscricao}";
                            $resultApi = @file_get_contents($urlApi);
                            $jsonApi   = json_decode($resultApi, true);

                            if (isset($jsonApi['status']) && $jsonApi['status'] === 'success' && !empty($jsonApi['data'][0])) {
                                $apiData = (object)$jsonApi['data'][0];
                                $formData['tipo']   = $apiData->tipo   ?? '';
                                $formData['padrao'] = $apiData->padrao ?? '';
                                $formData['valor']  = number_format((float)($apiData->valor_estimado ?? 0), 2, ',', '.');
                                $formData['id']     = $apiData->id     ?? '';
                            }
                        }
                    } else {
                        new TMessage('warning', "Feição sem propriedades para a inscrição {$inscricao}.");
                    }
                }
            } catch (Exception $e) {
                new TMessage('error', 'Erro ao consultar WFS: ' . htmlspecialchars($e->getMessage()));
            }

            // aplica no formulário (mesmo que parcial)
            $this->form->setData((object)$formData);
        }

        // Container visual
        $vbox = new TVBox;
        $vbox->style = 'width: 100%';
        $vbox->add($this->form);
        parent::add($vbox);
    }

    public function onCalcular($params)
    {
        $data = $this->form->getData();

        $area_unida = (float) str_replace(',', '.', (string)$data->area_unida);
        $zoneamento = $data->zoneamento ?? 'ZR-1';
        $tipo       = $data->tipo       ?? 'residencial';
        $padrao     = $data->padrao     ?? 'medio';

        $valoresZona = [
            'ZR-1' => 2800,
            'ZR-2' => 2000,
            'ZR-3' => 1500,
            'ZR-4' => 1200,
            'ZR-5' => 1000
        ];

        $multiplicadorTipo = [
            'residencial'   => 1.0,
            'comercial'     => 1.2,
            'industrial'    => 1.5,
            'institucional' => 1.1
        ];

        $multiplicadorPadrao = [
            'baixo' => 0.8,
            'medio' => 1.0,
            'alto'  => 1.3,
            'luxo'  => 1.6
        ];

        $valorM2   = $valoresZona[$zoneamento]     ?? 1000;
        $fatorTipo = $multiplicadorTipo[$tipo]     ?? 1.0;
        $fatorPad  = $multiplicadorPadrao[$padrao] ?? 1.0;

        $valorEstimado = $area_unida * $valorM2 * $fatorTipo * $fatorPad;
        $data->valor   = number_format($valorEstimado, 2, ',', '.');

        $this->form->setData($data);
    }

    public function onSave($params)
    {
        try {
            $data = $this->form->getData();

            $avaliacao = [
                'id'             => $data->id,
                'inscricao'      => $data->code,
                'zoneamento'     => $data->zoneamento,
                'area_unida'     => str_replace(',', '.', (string)$data->area_unida),
                'tipo'           => $data->tipo,
                'padrao'         => $data->padrao,
                'valor_estimado' => str_replace(['.', ','], ['', '.'], (string)$data->valor),
                'lote'           => $data->lote,
                'quadra'         => $data->quadra,
                'setor'          => $data->setor,
                'numero_lot'     => $data->numero_lot,
                'logradouro'     => $data->logradouro,
                'bairro'         => $data->bairro,
                'name'           => $data->name,
                'latitude'       => $data->latitude,
                'longitude'      => $data->longitude,
                'usuario_id'     => TSession::getValue('userid')
            ];

            $url = 'https://geo-el02.cloud.sisprom-br.com.br/vendor/geojequie/salvar_avaliacao.php';

            $ch = curl_init($url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 15);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($avaliacao));
            $result   = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlErr  = curl_error($ch);
            curl_close($ch);

            if ($result === false) {
                throw new Exception('Erro CURL: ' . $curlErr);
            }
            if ($httpCode !== 200) {
                throw new Exception("Erro HTTP na API: código $httpCode");
            }

            $response = json_decode($result, true);
            if (is_array($response) && ($response['status'] ?? '') === 'success') {
                new TMessage('info', 'Avaliação enviada e salva com sucesso!');
            } else {
                $msg = $response['message'] ?? 'Resposta inválida da API';
                throw new Exception($msg);
            }
        } catch (Exception $e) {
            new TMessage('error', 'Erro ao salvar via API: ' . $e->getMessage());
        }

        // Log do envio para análise posterior
        if (!is_dir(__DIR__)) {
            @mkdir(__DIR__, 0775, true);
        }
        file_put_contents(__DIR__ . '/log_envio_avaliacao.json', json_encode($avaliacao ?? [], JSON_PRETTY_PRINT));
    }

    public function onClose($param)
    {
        TScript::create("Template.closeRightPanel()");
    }

   public function onGeneratePdf($params)
{
    try {
        // 1) Sobe o limite de memória só para este trecho (se o provedor permitir)
        @ini_set('memory_limit', '1024M'); // 1 GB
        @ini_set('max_execution_time', '60');

        $data = $this->form->getData();

        // 2) HTML enxuto (sem imagens externas) e com fonte padrão leve
        $html = "
            <html>
            <head>
                <meta charset='utf-8'>
                <style>
                    body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 12px; }
                    h2 { margin: 0 0 10px 0; }
                    table { border-collapse: collapse; width: 100%; }
                    th, td { border: 1px solid #999; padding: 6px; vertical-align: top; }
                    th { width: 35%; text-align: left; background: #f4f4f4; }
                </style>
            </head>
            <body>
                <h2>Avaliação Imobiliária</h2>
                <table>
                    <tr><th>Inscrição</th><td>".htmlspecialchars((string)$data->code)."</td></tr>
                    <tr><th>Proprietário</th><td>".htmlspecialchars((string)$data->name)."</td></tr>
                    <tr><th>Zoneamento</th><td>".htmlspecialchars((string)$data->zoneamento)."</td></tr>
                    <tr><th>Área construída (m²)</th><td>".htmlspecialchars((string)$data->area_unida)."</td></tr>
                    <tr><th>Tipo de uso</th><td>".htmlspecialchars((string)$data->tipo)."</td></tr>
                    <tr><th>Padrão construtivo</th><td>".htmlspecialchars((string)$data->padrao)."</td></tr>
                    <tr><th>Valor estimado (R$)</th><td>".htmlspecialchars((string)$data->valor)."</td></tr>
                    <tr><th>Endereço</th><td>".htmlspecialchars((string)$data->logradouro).", ".htmlspecialchars((string)$data->bairro)."</td></tr>
                    <tr><th>Localização</th><td>Lat: ".htmlspecialchars((string)$data->latitude)." | Long: ".htmlspecialchars((string)$data->longitude)."</td></tr>
                </table>
            </body>
            </html>
        ";

        // 3) Opções que reduzem consumo
        $options = new Options();
        $options->setIsRemoteEnabled(false);              // desativa recursos remotos
        $options->setIsHtml5ParserEnabled(true);          // parser mais eficiente
        $options->setIsFontSubsettingEnabled(true);       // incorpora apenas subset de fontes
        $options->setDefaultFont('DejaVu Sans');          // evita fontes pesadas
        $options->setChroot(__DIR__ . '/../../..');       // sandbox (ajuste raiz do app)
        $options->setTempDir(sys_get_temp_dir());         // temp do SO (ou defina uma pasta sua)
        $options->setDpi(96);                             // DPI menor, menos memória

        $dompdf = new Dompdf($options);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->loadHtml($html);

        // 4) Render
        $dompdf->render();

        // 5) Salva em tmp/ garantindo pasta e nome seguro
        $dir = 'tmp';
        if (!is_dir($dir)) { @mkdir($dir, 0775, true); }

        $safeCode = preg_replace('/\W+/', '', (string)$data->code);
        $filePath = $dir . '/avaliacao_' . $safeCode . '.pdf';

        // stream para arquivo (menos pico de memória do que manter em string por muito tempo)
        $pdf = $dompdf->output();
        file_put_contents($filePath, $pdf);
        unset($pdf); // libera memória

        if (!file_exists($filePath)) {
            throw new Exception("Erro ao gerar PDF: arquivo não encontrado em {$filePath}");
        }

        // 6) Exibe no TWindow
        $window = new TWindow('Avaliação Imobiliária - PDF');
        $window->setSize(0.8, 0.8);

        $object = new TElement('object');
        $object->data  = $filePath;
        $object->type  = 'application/pdf';
        $object->style = "width: 100%; height: calc(100% - 10px)";
        $object->add('Seu navegador não suporta PDF embutido. <a href="'.$filePath.'" target="_blank">Clique aqui para baixar</a>.');

        $window->add($object);
        $window->show();

    } catch (Exception $e) {
        new TMessage('error', 'Erro ao gerar PDF: ' . $e->getMessage());
    }
}

}
