<?php

class KanbanView extends TPage
{
    private $kanban;
    private $container;

    public function __construct()
    {
        parent::__construct();

        TPage::include_css('app/resources/kanban/custom-kanban.css');

        $this->kanban = new TKanban;
        $this->kanban->setStageHeight('80vh');

        $this->loadKanban();

        $vbox = new TVBox;
        $vbox->style = 'width: 100%';
        $vbox->add(new TXMLBreadCrumb('menu.xml', __CLASS__));

        $containerKanban = new TElement('div');
        $containerKanban->id = 'kanban_container';
        $containerKanban->add($this->kanban);

        $vbox->add($containerKanban);

        // Container geral da página
        $this->container = new TElement('div');
        $this->container->id = 'kanban_view_container';
        $this->container->add($vbox);

        parent::add($this->container);

        // Funções JS para recarregar conteúdos via AJAX
        TScript::create("
            function loadKanban() {
                TLoader.loadContent('kanban_container', 'engine.php?class=KanbanView&method=loadKanbanAjax');
            }

            function reloadKanbanView() {
                TLoader.loadContent('kanban_view_container', 'engine.php?class=KanbanView&method=onReload');
            }
        ");

        // Script swipe para o kanban
        TScript::create("
    setTimeout(function() {
        var container = document.querySelector('.tkanban-container');

        if (container) {
            var hammertime = new Hammer(container);
            hammertime.get('swipe').set({ direction: Hammer.DIRECTION_HORIZONTAL });

            hammertime.on('swipeleft', function() {
                container.scrollBy({ left: 300, behavior: 'smooth' });
            });

            hammertime.on('swiperight', function() {
                container.scrollBy({ left: -300, behavior: 'smooth' });
            });
        }
    }, 500);
");
    }

    private function loadKanban()
    {
        $this->kanban->clear();

        $stage_ids = [
            'A Fazer' => 1,
            'Em Andamento' => 2,
            'Concluído' => 3
        ];

        $this->kanban->addStage(1, '<i class="fas fa-list"></i> A Fazer');
        $this->kanban->addStage(2, '<i class="fas fa-spinner"></i> Em Andamento');
        $this->kanban->addStage(3, '<i class="fas fa-check-circle"></i> Concluído');

        $user_id = TSession::getValue('userid');
        $group_ids = TSession::getValue('usergroupids') ?? [];
        if (!is_array($group_ids)) $group_ids = [$group_ids];

        TTransaction::open('permission');
        $criteria = new TCriteria();
        if (!in_array(1, $group_ids)) {
            $criteria->add(new TFilter('assigned_to', '=', $user_id));
        }

        $tasks = Kanban::getObjects($criteria);
        TTransaction::close();

        $colors = [
            'A Fazer' => '#FF6B6B',
            'Em Andamento' => '#FFD93D',
            'Concluído' => '#57D557'
        ];

        TTransaction::open('permission');

        foreach ($tasks as $task) {
            $desc = $task->description ?? '';
            if (strlen($desc) > 47) $desc = substr($desc, 0, 47) . '...';

            $color = $colors[$task->status ?? ''] ?? '#AAAAAA';
            $stage_id = $stage_ids[$task->status ?? ''] ?? 1;

            $due_date = $task->due_date ? new DateTime($task->due_date) : null;
            $now = new DateTime();

            // Buscar nome do responsável
            $responsavel = 'Desconhecido';
            if (!empty($task->assigned_to)) {
                $user = SystemUser::find($task->assigned_to);
                if ($user) {
                    $responsavel = $user->name;
                }
            }

            // Buscar nome do criador
            $criador = 'Desconhecido';
            if (!empty($task->created_by)) {
                $user_creator = SystemUser::find($task->created_by);
                if ($user_creator) {
                    $criador = $user_creator->name;
                }
            }

             $label = '<div class="tkanban-item-content" style="border-left-color:' . $color . '">'
                . '<strong>' . htmlspecialchars($task->title ?? '') . '</strong><br>'
                . '<span style="color: #555;">' . htmlspecialchars($desc) . '</span><br>'
                . '<span style="color: #555;">👤 Responsável: ' . htmlspecialchars($responsavel) . '</small><br>'
                . '<span style="color: #555;">👤 Criado por: ' . htmlspecialchars($criador) . '</small><br>';

            if ($due_date) {
                $due_date_str = $due_date->format('d/m/Y');
                $due_date_plus_one = clone $due_date;
                $due_date_plus_one->modify('+1 day');

                if ($now > $due_date_plus_one && ($task->status ?? '') !== 'Concluído') {
                    $label .= '<small style="color: red; font-weight: bold;">⚠️ Atrasado! Prazo: ' . $due_date_str . '</small>';
                } else {
                    $label .= '<small style="color: #888;">📅 Prazo: ' . $due_date_str . '</small>';
                }
            }

            $label .= '</div>';

            $this->kanban->addItem(
                $task->id,
                $stage_id,
                $label,
                '',
                $color
            );
        }

        TTransaction::close();

        $this->kanban->addItemAction('Editar', new TAction([$this, 'onItemEdit']), 'far:edit blue');

        // Exibir botão "Excluir" somente para administradores (grupo 1)
        if (in_array(1, $group_ids)) {
            $this->kanban->addItemAction('Excluir', new TAction([$this, 'onItemDelete']), 'far:trash-alt red');
        }

        $this->kanban->addItemAction('Visualizar', new TAction([$this, 'onItemView']), 'fa:eye green');

        $this->kanban->setItemDropAction(new TAction([$this, 'onUpdateItemDrop']));
    }

    public static function onItemEdit($param)
    {
        try {
            TTransaction::open('permission');
            $task = new Kanban($param['id']);
            TTransaction::close();

            $form = new BootstrapFormBuilder('form_edit_task');
            
            $id    = new TEntry('id');
            $title = new TEntry('title');
            $description = new TText('description');
            $status = new TCombo('status');
            $observation = new TText('observation'); // novo campo observação

            $status->addItems([
                'A Fazer' => 'A Fazer',
                'Em Andamento' => 'Em Andamento',
                'Concluído' => 'Concluído'
            ]);

            // Campos não editáveis
            $id->setEditable(false);
            $title->setEditable(false);
            $description->setEditable(false);

            if (!empty($task->filename)) {
                $url = "engine.php?class=KanbanView&method=onDownload&id={$task->id}";
                $link = new THyperLink('Download do Anexo', $url, 'blue', 12, 'fa:file');
                $link->{'target'} = '_blank';  // define target assim
                $form->addFields([new TLabel('Arquivo')], [$link]);
            }

            $form->addFields([new TLabel('ID')], [$id]);
            $form->addFields([new TLabel('Título')], [$title]);
            $form->addFields([new TLabel('Descrição')], [$description]);
            $form->addFields([new TLabel('Status')], [$status]);
            $form->addFields([new TLabel('Observação')], [$observation]);

            $form->setData($task);

            $form->addAction('Salvar', new TAction([__CLASS__, 'onSaveEdit']), 'fa:save green');

            $window = TWindow::create('Editar Tarefa', 600, 400);
            $window->add($form);
            $window->show();
        } catch (Exception $e) {
            new TMessage('error', 'Erro ao carregar tarefa: ' . $e->getMessage());
            TTransaction::rollback();
        }
    }

    public static function onItemDelete($param)
    {
        if (!empty($param['confirm']) && $param['confirm'] === 'yes') {
            try {
                TTransaction::open('permission');
                $task = new Kanban($param['id']);
                $task->delete();
                TTransaction::close();

                new TMessage('info', 'Tarefa excluída com sucesso!');
                // Para atualizar visualmente, pode recarregar a página ou implementar reload parcial

                TScript::create('window.location.reload();');

            } catch (Exception $e) {
                new TMessage('error', 'Erro ao excluir tarefa: ' . $e->getMessage());
                TTransaction::rollback();
            }
        } else {
            $action = new TAction([__CLASS__, 'onItemDelete']);
            $action->setParameters(['id' => $param['id'], 'confirm' => 'yes']);
            new TQuestion('Deseja realmente excluir esta tarefa?', $action);
        }
    }

    public static function onUpdateItemDrop($param)
    {
        $stage_map = [
            1 => 'A Fazer',
            2 => 'Em Andamento',
            3 => 'Concluído'
        ];

        try {
            TTransaction::open('permission');
            $task = new Kanban($param['id']);
            $new_status = $stage_map[$param['stage_id']] ?? $task->status;
            $task->status = $new_status;
            $task->store();
            TTransaction::close();

            new TMessage('info', 'Status atualizado com sucesso!');
        } catch (Exception $e) {
            new TMessage('error', $e->getMessage());
            TTransaction::rollback();
        }
    }

    public static function onSaveEdit($param)
    {
        try {
            TTransaction::open('permission');

            $task = new Kanban($param['id']);
            $task->title = $param['title'];
            $task->description = $param['description'];
            $task->status = $param['status'];
            $task->observation = $param['observation'];
            $task->store();

            TTransaction::close();

        new TMessage('info', 'Tarefa atualizada com sucesso!');
        TScript::create('window.location.reload();');

    } catch (Exception $e) {
        new TMessage('error', 'Erro ao salvar: ' . $e->getMessage());
        TTransaction::rollback();
    }
}

    public static function onItemView($param)
    {
        try {
            TTransaction::open('permission');
            $task = new Kanban($param['id']);
            TTransaction::close();

            $box = new TElement('div');
            $box->style = 'padding: 10px; font-family: Arial;';

            $box->add('<h3>' . htmlspecialchars($task->title ?? '') . '</h3>');
            $box->add('<p><strong>Status:</strong> ' . htmlspecialchars($task->status ?? '') . '</p>');

            if (!empty($task->due_date)) {
                $due_date = new DateTime($task->due_date);
                $now = new DateTime();
                $due_date_str = $due_date->format('d/m/Y');
                $due_date_plus_one = clone $due_date;
                $due_date_plus_one->modify('+1 day');

                if ($now > $due_date_plus_one && ($task->status ?? '') !== 'Concluído') {
                    $box->add('<p><strong>Prazo:</strong> <span style="color: red; font-weight: bold;">⚠️ Atrasado! Prazo: ' . $due_date_str . '</span></p>');
                } else {
                    $box->add('<p><strong>Prazo:</strong> <span style="color: #555;">📅 ' . $due_date_str . '</span></p>');
                }
            }

            $box->add('<p><strong>Descrição:</strong><br>' . nl2br(htmlspecialchars($task->description ?? '')) . '</p>');

            if (!empty($task->files)) {
                $urlDownload = "engine.php?class=KanbanView&method=onDownload&id={$task->id}";
                $box->add('<p><strong>Anexo:</strong> <a href="' . $urlDownload . '" target="_blank" style="color: blue; text-decoration: underline;">Download do arquivo</a></p>');
            }

            $window = TWindow::create('Detalhes da Tarefa', 500, 350);
            $window->add($box);
            $window->show();
        } catch (Exception $e) {
            new TMessage('error', 'Erro ao exibir tarefa: ' . $e->getMessage());
            TTransaction::rollback();
        }
    }

    public static function onDownload($param)
    {
        try {
            TTransaction::open('permission');

            $id = $param['id'] ?? null;
            if (!$id) {
                throw new Exception('ID da tarefa não informado.');
            }

            $task = new Kanban($id);

            if (empty($task->filename)) {
                throw new Exception('Nenhum arquivo anexado para esta tarefa.');
            }

            $file_path = "files/kanban/{$id}/{$task->filename}";

            if (!file_exists($file_path)) {
                throw new Exception('Arquivo não encontrado: ' . $file_path);
            }

            // Limpa todos os buffers de saída
            while (ob_get_level()) {
                ob_end_clean();
            }

            header('Content-Description: File Transfer');
            header('Content-Type: application/octet-stream');
            header('Content-Disposition: attachment; filename="' . basename($file_path) . '"');
            header('Expires: 0');
            header('Cache-Control: must-revalidate');
            header('Pragma: public');
            header('Content-Length: ' . filesize($file_path));

            readfile($file_path);
            exit;

        } catch (Exception $e) {
            TTransaction::rollback();
            new TMessage('error', $e->getMessage());
        }
    }

    public static function loadKanbanAjax()
    {
        try {
            // Cria uma instância para montar o kanban
            $page = new KanbanView;
            $page->loadKanban();

            // Renderiza somente o container do kanban
            $container = new TElement('div');
            $container->id = 'kanban_container';
            $container->add($page->kanban);

            echo $container->getContents();

        } catch (Exception $e) {
            echo '<div class="alert alert-danger">Erro ao carregar Kanban: ' . $e->getMessage() . '</div>';
        }
    }

    public static function onReload()
    {
        $page = new KanbanView;
        echo $page->container->getContents();
    }
}
