<?php
use Adianti\Control\TAction;
use Adianti\Control\TPage;
use Adianti\Widget\Container\TPanelGroup;
use Adianti\Widget\Dialog\TMessage;
use Adianti\Widget\Form\TEntry;
use Adianti\Database\TRepository;
use Adianti\Database\TCriteria;
use Adianti\Database\TFilter;
use Adianti\Widget\Wrapper\TQuickForm;
use Adianti\Widget\Datagrid\TDataGridColumn;
use Adianti\Widget\Datagrid\TDataGridAction;
use Adianti\Widget\Datagrid\TDataGrid;
use Adianti\Widget\Dialog\TQuestion;

use Adianti\Database\TTransaction;

// Compat: alias para TSession em diferentes versões
if (!class_exists('TSession')) {
    if (class_exists('\Adianti\Core\TSession')) {
        class_alias('\Adianti\Core\TSession', 'TSession');
    } elseif (class_exists('\Adianti\Registry\TSession')) {
        class_alias('\Adianti\Registry\TSession', 'TSession');
    }
}

class BairroList extends TPage
{
    private $form;
    private $datagrid;
    private $loaded;

    public function __construct()
    {
        parent::__construct();

        // Formulário de busca
        $this->form = new TQuickForm('form_search_Bairro');
        $this->form->setFormTitle('Bairro - Busca');

        $id   = new TEntry('id');
        $nome = new TEntry('nome');

        $this->form->addQuickField('ID',   $id,   100);
        $this->form->addQuickField('Nome', $nome, 300);

        $this->form->setData( TSession::getValue('search_Bairro') );

        $this->form->addQuickAction('Buscar', new TAction([$this, 'onSearch']), 'fa:search');
        $this->form->addQuickAction('Novo',   new TAction(['BairroForm', 'onClear']), 'fa:plus green');

        // Grid
        $this->datagrid = new TDataGrid;
        $this->datagrid->style     = 'width: 100%';
        $this->datagrid->datatable = 'true';

        $col_id             = new TDataGridColumn('id', 'ID', 'center', '10%');
        $col_municipio_nome = new TDataGridColumn('municipio_nome', 'Município/Localidade', 'left', '30%');
        $col_nome           = new TDataGridColumn('nome', 'Nome', 'left', '60%');

        $this->datagrid->addColumn($col_id);
        $this->datagrid->addColumn($col_municipio_nome);
        $this->datagrid->addColumn($col_nome);

        // Ações (agora com setField('id'))
        $action_edit = new TDataGridAction(['BairroForm', 'onEdit']);
        $action_edit->setLabel('Editar');
        $action_edit->setImage('fa:edit blue');
        $action_edit->setField('id');

        $action_del  = new TDataGridAction([$this, 'onDelete']);
        $action_del->setLabel('Excluir');
        $action_del->setImage('fa:trash red');
        $action_del->setField('id');

        $this->datagrid->addAction($action_edit);
        $this->datagrid->addAction($action_del);

        $this->datagrid->createModel();

        // Painel
        $panel = new TPanelGroup('Bairro - Lista');
        $panel->add($this->form);
        $panel->add($this->datagrid);

        parent::add($panel);
    }

    public function onSearch($param)
    {
        $data = $this->form->getData();
        TSession::setValue('search_Bairro', $data);
        $this->onReload($param);
    }

    public function onReload($param = null)
    {
        try {
            TTransaction::open('comum');

            $repository = new TRepository('Bairro');
            $criteria   = new TCriteria;

            $data = TSession::getValue('search_Bairro');

            if ($data) {
                if (!empty($data->id)) {
                    $criteria->add(new TFilter('id', '=', $data->id));
                }
                if (!empty($data->nome)) {
                    $criteria->add(new TFilter('nome', 'ilike', "%{$data->nome}%"));
                }
            }

            $objects = $repository->load($criteria);

            // Preload nomes dos municípios em lote
            $municipioMap = [];
            if ($objects) {
                $ids = [];
                foreach ($objects as $o) {
                    if (!empty($o->id_munic_local)) {
                        $ids[] = (int) $o->id_munic_local;
                    }
                }
                if ($ids) {
                    $ids = array_values(array_unique($ids));
                    $repoMun = new TRepository('MunicipioLocalidade');
                    $critMun = new TCriteria;
                    // você pode passar array direto no IN
                    $critMun->add(new TFilter('id', 'in', $ids));
                    $munObjs = $repoMun->load($critMun);
                    if ($munObjs) {
                        foreach ($munObjs as $m) {
                            $municipioMap[(int)$m->id] = $m->nome;
                        }
                    }
                }
            }

            $this->datagrid->clear();
            if ($objects) {
                foreach ($objects as $object) {
                    $object->municipio_nome = $municipioMap[(int)$object->id_munic_local] ?? '';
                    $this->datagrid->addItem($object);
                }
            }

            TTransaction::close();
            $this->loaded = true;
        } catch (Exception $e) {
            new TMessage('error', $e->getMessage());
            TTransaction::rollback();
        }
    }

    public function onDelete($param)
    {
        $action = new TAction([$this, 'Delete']);
        $action->setParameters($param);
        new TQuestion('Confirma excluir?', $action);
    }

    public function Delete($param)
    {
        try {
            TTransaction::open('comum');
            $object = new Bairro($param['id']);
            $object->delete();
            TTransaction::close();
            $this->onReload();
            new TMessage('info', 'Registro excluído');
        } catch (Exception $e) {
            new TMessage('error', $e->getMessage());
            TTransaction::rollback();
        }
    }

    public function show()
    {
        if (!$this->loaded) {
            $this->onReload(func_get_arg(0));
        }
        parent::show();
    }
}
